###LOAD DEPENDENT PACKAGES
library(xlsx)
library(rstatix)
library(outliers)
library(ggplot2)
library(ggpubr)

###FORMAT DATA
Data <- read.xlsx("Discrete_Kinematics.xlsx", sheetIndex = 1)
Data[,4:23] <- abs(Data[,4:23])
Ankle <- Data[,c(1,14:18)]

###OUTLIER IDENTIFICATION

Ay.Ankle <- Ankle[1:30,]
Ma.Ankle <- Ankle[31:60,]
Ir.Ankle <- Ankle[61:90,]

#Grubb's test to identify outliers
#Aylesburys
for(i in 2:ncol(Ay.Ankle)){
  DV <- names(Ay.Ankle[i])
  res.grub <- grubbs.test(Ay.Ankle[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}
#Mallards
for(i in 2:ncol(Ma.Ankle)){
  DV <- names(Ma.Ankle[i])
  res.grub <- grubbs.test(Ma.Ankle[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}
#Runners
for(i in 2:ncol(Ir.Ankle)){
  DV <- names(Ir.Ankle[i])
  res.grub <- grubbs.test(Ir.Ankle[,i], type = 11) #test if highest and lowest values are outliers
  print(DV)
  print(res.grub)
}

#Boxplots to investigation data distribution

Box.Ankle.Max <- ggplot(Ankle, aes(x=Breed, y=Ankle_Max_Flex, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Maximum Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Ankle.Min <- ggplot(Ankle, aes(x=Breed, y=Ankle_Min_Flex, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Minimum Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Toe.On <- ggplot(Ankle, aes(x=Breed, y=Ankle_Toe_on, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-on Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Toe.Off <- ggplot(Ankle, aes(x=Breed, y=Ankle_Toe_off, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-off Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Box.Mid.Stance <- ggplot(Ankle, aes(x=Breed, y=Ankle_Mid_Stance, fill=Breed)) + 
  geom_boxplot(width=0.8, lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Mid-stance Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

#combine and publish violin plots
tiff(filename = "Boxplot_Ankle.tif", width = 166, height = 166, units = "mm", res = 600, compression = "lzw") #166x166*2 or 166x166*1.5

plot.new()

ggarrange(Box.Ankle.Max, 
          Box.Ankle.Min, 
          Box.Toe.On, 
          Box.Toe.Off, 
          Box.Mid.Stance, 
          labels = c("A", "B", "C", "D", "E"),
          hjust = -4,
          ncol = 2, nrow = 3)

dev.off()

###MAIN ANALYSIS

#Seperate variables to individual columns (required as outliers are omitted via row deletion)
Ankle.Max.Flex.Data <- Ankle[,1:2]
Ankle.Min.Flex.Data <- Ankle[,c(1,3)]
Ankle.Toe.On.Data <- Ankle[,c(1,4)]
Ankle.Toe.Off.Data <- Ankle[,c(1,5)]
Ankle.Mid.Stance.Data <- Ankle[,c(1,6)]

#Omit outliers (currently performed manually)
Ankle.Toe.Off.Data <- Ankle.Toe.Off.Data[!(row.names(Ankle.Toe.Off.Data) %in% c("10")),]

#Shapiro-Wilks Test for Normality
res.shap <- shapiro_test(Ankle.Min.Flex.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Ankle.Max.Flex.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Ankle.Toe.On.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Ankle.Toe.Off.Data[,2])
print(res.shap)
res.shap <- shapiro_test(Ankle.Mid.Stance.Data[,2])
print(res.shap)

#Levene's test for homogeneity of variance
res.lt <- levene_test(Ankle.Min.Flex.Data[,2] ~ Breed, data = Ankle.Min.Flex.Data)
print(res.lt)
res.lt <- levene_test(Ankle.Max.Flex.Data[,2] ~ Breed, data = Ankle.Max.Flex.Data)
print(res.lt)
res.lt <- levene_test(Ankle.Toe.On.Data[,2] ~ Breed, data = Ankle.Toe.On.Data)
print(res.lt)
res.lt <- levene_test(Ankle.Toe.Off.Data[,2] ~ Breed, data = Ankle.Toe.Off.Data)
print(res.lt)
res.lt <- levene_test(Ankle.Mid.Stance.Data[,2] ~ Breed, data = Ankle.Mid.Stance.Data)
print(res.lt)

#Kruskal-Wallis Test & output to .xls
KW.Ankle.Min.Flex <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Ankle.Min.Flex <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Ankle.Min.Flex.Data[,2] ~ Breed, data = Ankle.Min.Flex.Data)
res.ES <- kruskal_effsize(Ankle.Min.Flex.Data[,2] ~ Breed, data = Ankle.Min.Flex.Data)
print(res.KW)
print(res.ES)
KW.Ankle.Min.Flex[i,] <- data.frame(var = names(Ankle.Min.Flex.Data[2]), res.KW, row.names = NULL)
ES.Ankle.Min.Flex[i,] <- data.frame(var = names(Ankle.Min.Flex.Data[2]), res.ES, row.names = NULL)
colnames(KW.Ankle.Min.Flex) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Ankle.Min.Flex) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Ankle.Min.Flex <- cbind(KW.Ankle.Min.Flex, ES.Ankle.Min.Flex)
Final.Ankle.Min.Flex <- Final.Ankle.Min.Flex[ -c(2, 7:10, 12:13) ]
Final.Ankle.Min.Flex <- Final.Ankle.Min.Flex[ -c(1:5), ]

KW.Ankle.Max.Flex <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Ankle.Max.Flex <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Ankle.Max.Flex.Data[,2] ~ Breed, data = Ankle.Max.Flex.Data)
res.ES <- kruskal_effsize(Ankle.Max.Flex.Data[,2] ~ Breed, data = Ankle.Max.Flex.Data)
print(res.KW)
print(res.ES)
KW.Ankle.Max.Flex[i,] <- data.frame(var = names(Ankle.Max.Flex.Data[2]), res.KW, row.names = NULL)
ES.Ankle.Max.Flex[i,] <- data.frame(var = names(Ankle.Max.Flex.Data[2]), res.ES, row.names = NULL)
colnames(KW.Ankle.Max.Flex) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Ankle.Max.Flex) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Ankle.Max.Flex <- cbind(KW.Ankle.Max.Flex, ES.Ankle.Max.Flex)
Final.Ankle.Max.Flex <- Final.Ankle.Max.Flex[ -c(2, 7:10, 12:13) ]
Final.Ankle.Max.Flex <- Final.Ankle.Max.Flex[ -c(1:5), ]

KW.Ankle.Toe.On <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Ankle.Toe.On <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Ankle.Toe.On.Data[,2] ~ Breed, data = Ankle.Toe.On.Data)
res.ES <- kruskal_effsize(Ankle.Toe.On.Data[,2] ~ Breed, data = Ankle.Toe.On.Data)
print(res.KW)
print(res.ES)
KW.Ankle.Toe.On[i,] <- data.frame(var = names(Ankle.Toe.On.Data[2]), res.KW, row.names = NULL)
ES.Ankle.Toe.On[i,] <- data.frame(var = names(Ankle.Toe.On.Data[2]), res.ES, row.names = NULL)
colnames(KW.Ankle.Toe.On) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Ankle.Toe.On) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Ankle.Toe.On <- cbind(KW.Ankle.Toe.On, ES.Ankle.Toe.On)
Final.Ankle.Toe.On <- Final.Ankle.Toe.On[ -c(2, 7:10, 12:13) ]
Final.Ankle.Toe.On <- Final.Ankle.Toe.On[ -c(1:5), ]

KW.Ankle.Toe.Off <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Ankle.Toe.Off <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Ankle.Toe.Off.Data[,2] ~ Breed, data = Ankle.Toe.Off.Data)
res.ES <- kruskal_effsize(Ankle.Toe.Off.Data[,2] ~ Breed, data = Ankle.Toe.Off.Data)
print(res.KW)
print(res.ES)
KW.Ankle.Toe.Off[i,] <- data.frame(var = names(Ankle.Toe.Off.Data[2]), res.KW, row.names = NULL)
ES.Ankle.Toe.Off[i,] <- data.frame(var = names(Ankle.Toe.Off.Data[2]), res.ES, row.names = NULL)
colnames(KW.Ankle.Toe.Off) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Ankle.Toe.Off) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Ankle.Toe.Off <- cbind(KW.Ankle.Toe.Off, ES.Ankle.Toe.Off)
Final.Ankle.Toe.Off <- Final.Ankle.Toe.Off[ -c(2, 7:10, 12:13) ]
Final.Ankle.Toe.Off <- Final.Ankle.Toe.Off[ -c(1:5), ]

KW.Ankle.Mid.Stance <- data.frame(matrix(nrow = 2, ncol = 7))
ES.Ankle.Mid.Stance <- data.frame(matrix(nrow = 2, ncol = 6))
res.KW <- kruskal_test(Ankle.Mid.Stance.Data[,2] ~ Breed, data = Ankle.Mid.Stance.Data)
res.ES <- kruskal_effsize(Ankle.Mid.Stance.Data[,2] ~ Breed, data = Ankle.Mid.Stance.Data)
print(res.KW)
print(res.ES)
KW.Ankle.Mid.Stance[i,] <- data.frame(var = names(Ankle.Mid.Stance.Data[2]), res.KW, row.names = NULL)
ES.Ankle.Mid.Stance[i,] <- data.frame(var = names(Ankle.Mid.Stance.Data[2]), res.ES, row.names = NULL)
colnames(KW.Ankle.Mid.Stance) <- c("Parameter", "y", "n", "statistic", "df", "p", "method")
colnames(ES.Ankle.Mid.Stance) <- c("Parameter", "y", "n", "effsize", "method", "magnitude")
Final.Ankle.Mid.Stance <- cbind(KW.Ankle.Mid.Stance, ES.Ankle.Mid.Stance)
Final.Ankle.Mid.Stance <- Final.Ankle.Mid.Stance[ -c(2, 7:10, 12:13) ]
Final.Ankle.Mid.Stance <- Final.Ankle.Mid.Stance[ -c(1:5), ]

write.xlsx(Final.Ankle.Min.Flex, file = "KW_Ankle.xlsx", sheetName = "Ankle.Min.Flex")
write.xlsx(Final.Ankle.Max.Flex, file = "KW_Ankle.xlsx", sheetName = "Ankle.Max.Flex", append = TRUE)
write.xlsx(Final.Ankle.Toe.On, file = "KW_Ankle.xlsx", sheetName = "Ankle.Toe.On", append = TRUE)
write.xlsx(Final.Ankle.Toe.Off, file = "KW_Ankle.xlsx", sheetName = "Ankle.Toe.Off", append = TRUE)
write.xlsx(Final.Ankle.Mid.Stance, file = "KW_Ankle.xlsx", sheetName = "Ankle.Mid.Stance", append = TRUE)

#Dunn's Test & output to .xls
dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Ankle.Min.Flex <- dunn_vector(Ankle.Min.Flex.Data, reformulate("Breed", names(Ankle.Min.Flex.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Ankle.Min.Flex

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Ankle.Max.Flex <- dunn_vector(Ankle.Max.Flex.Data, reformulate("Breed", names(Ankle.Max.Flex.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Ankle.Max.Flex

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Ankle.Toe.On <- dunn_vector(Ankle.Toe.On.Data, reformulate("Breed", names(Ankle.Toe.On.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Ankle.Toe.On

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Ankle.Toe.Off <- dunn_vector(Ankle.Toe.Off.Data, reformulate("Breed", names(Ankle.Toe.Off.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Ankle.Toe.Off

dunn_vector <- Vectorize(dunn_test, vectorize.args="formula", SIMPLIFY=F)
reformulate <- Vectorize(reformulate, vectorize.args="response")
DN.Ankle.Mid.Stance <- dunn_vector(Ankle.Mid.Stance.Data, reformulate("Breed", names(Ankle.Mid.Stance.Data)[2]), p.adjust.method="bonferroni") #2:15 for linear measurements 2:11 for hulls
DN.Ankle.Mid.Stance

write.xlsx(DN.Ankle.Min.Flex, file = "Dunn_Ankle.xlsx", sheetName = "Ankle.Min.Flex")
write.xlsx(DN.Ankle.Max.Flex, file = "Dunn_Ankle.xlsx", sheetName = "Ankle.Max.Flex", append = TRUE)
write.xlsx(DN.Ankle.Toe.On, file = "Dunn_Ankle.xlsx", sheetName = "Ankle.Toe.On", append = TRUE)
write.xlsx(DN.Ankle.Toe.Off, file = "Dunn_Ankle.xlsx", sheetName = "Ankle.Toe.Off", append = TRUE)
write.xlsx(DN.Ankle.Mid.Stance, file = "Dunn_Ankle.xlsx", sheetName = "Ankle.Mid.Stance", append = TRUE)

#Violin Plots
Violin.Ankle.Min.Flex <- ggplot(Ankle.Min.Flex.Data, aes(x=Breed, y=Ankle_Min_Flex, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Minimum Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Ankle.Max.Flex <- ggplot(Ankle.Max.Flex.Data, aes(x=Breed, y=Ankle_Max_Flex, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Maximum Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Ankle.Toe.On <- ggplot(Ankle.Toe.On.Data, aes(x=Breed, y=Ankle_Toe_on, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-on Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Ankle.Toe.Off <- ggplot(Ankle.Toe.Off.Data, aes(x=Breed, y=Ankle_Toe_off, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Toe-off Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

Violin.Ankle.Mid.Stance <- ggplot(Ankle.Mid.Stance.Data, aes(x=Breed, y=Ankle_Mid_Stance, fill=Breed)) + 
  geom_violin(trim = TRUE, lwd=1) +
  geom_boxplot(width=0.05, fill="white", lwd=1) +
  scale_fill_manual(values=c("sienna3", "skyblue4", "lightyellow2")) +
  labs(x = "Breed", y = "Mid-stance Ankle Flexion (\u00B0)") +
  theme(legend.position="none") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),panel.background = element_blank(), axis.line = element_line(colour = "black"))

#combine and publish violin plots
tiff(filename = "Violin_Ankle.tif", width = 166, height = 166, units = "mm", res = 600, compression = "lzw") #166x166*2 or 166x166*1.5

plot.new()

ggarrange(Violin.Ankle.Min.Flex, 
          Violin.Ankle.Max.Flex, 
          Violin.Ankle.Toe.On, 
          Violin.Ankle.Toe.Off, 
          Violin.Ankle.Mid.Stance, 
          labels = c("A", "B", "C", "D", "E"),
          hjust = -4,
          ncol = 2, nrow = 3)

dev.off()